-- database_schema.sql v1.0
-- Betatester.it - AI Challenge Arena
-- Execute this in phpMyAdmin

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- --------------------------------------------------------
-- Table: admin_users (authentication)
-- --------------------------------------------------------
DROP TABLE IF EXISTS `admin_users`;
CREATE TABLE `admin_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(50) NOT NULL,
  `password_hash` varchar(255) NOT NULL,
  `email` varchar(100) DEFAULT NULL,
  `last_login` datetime DEFAULT NULL,
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `username` (`username`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Default admin user (password: admin123 - CHANGE THIS!)
INSERT INTO `admin_users` (`username`, `password_hash`, `email`) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@betatester.it');

-- --------------------------------------------------------
-- Table: ai_models
-- --------------------------------------------------------
DROP TABLE IF EXISTS `ai_models`;
CREATE TABLE `ai_models` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(100) NOT NULL,
  `provider` varchar(50) NOT NULL COMMENT 'anthropic, openai, google, xai, deepseek',
  `model_code` varchar(100) NOT NULL COMMENT 'API model identifier',
  `api_key` text DEFAULT NULL,
  `api_endpoint` varchar(255) DEFAULT NULL,
  `is_active` tinyint(1) NOT NULL DEFAULT 1,
  `can_generate` tinyint(1) NOT NULL DEFAULT 1,
  `can_vote` tinyint(1) NOT NULL DEFAULT 1,
  `display_order` int(11) NOT NULL DEFAULT 0,
  `color_hex` varchar(7) DEFAULT '#666666' COMMENT 'Brand color for charts',
  `icon_url` varchar(255) DEFAULT NULL,
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  `updated_at` datetime DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_provider` (`provider`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Pre-populated AI models (API keys to be added)
INSERT INTO `ai_models` (`name`, `provider`, `model_code`, `api_endpoint`, `display_order`, `color_hex`) VALUES
('Claude Sonnet 4', 'anthropic', 'claude-sonnet-4-20250514', 'https://api.anthropic.com/v1/messages', 1, '#D97706'),
('GPT-4o', 'openai', 'gpt-4o', 'https://api.openai.com/v1/chat/completions', 2, '#10A37F'),
('Gemini 2.5 Pro', 'google', 'gemini-2.5-pro-preview-06-05', 'https://generativelanguage.googleapis.com/v1beta/models/', 3, '#4285F4'),
('Grok 3', 'xai', 'grok-3', 'https://api.x.ai/v1/chat/completions', 4, '#000000'),
('DeepSeek V3', 'deepseek', 'deepseek-chat', 'https://api.deepseek.com/v1/chat/completions', 5, '#0066FF');

-- --------------------------------------------------------
-- Table: challenges
-- --------------------------------------------------------
DROP TABLE IF EXISTS `challenges`;
CREATE TABLE `challenges` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `prompt_text` text NOT NULL,
  `slug` varchar(255) DEFAULT NULL,
  `title` varchar(255) DEFAULT NULL COMMENT 'Human readable title',
  `description` text DEFAULT NULL COMMENT 'SEO description',
  `status` enum('draft','generating','voting','completed','error') NOT NULL DEFAULT 'draft',
  `is_published` tinyint(1) NOT NULL DEFAULT 0,
  `error_message` text DEFAULT NULL,
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  `completed_at` datetime DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `slug` (`slug`),
  KEY `idx_status` (`status`),
  KEY `idx_published` (`is_published`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: challenge_outputs
-- --------------------------------------------------------
DROP TABLE IF EXISTS `challenge_outputs`;
CREATE TABLE `challenge_outputs` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `challenge_id` int(11) NOT NULL,
  `ai_model_id` int(11) NOT NULL,
  `output_html` longtext DEFAULT NULL,
  `file_path` varchar(255) DEFAULT NULL COMMENT 'Relative path: /slug/1.html',
  `file_number` int(11) DEFAULT NULL COMMENT 'Randomized: 1,2,3...',
  `generation_time_ms` int(11) DEFAULT NULL,
  `tokens_used` int(11) DEFAULT NULL,
  `status` enum('pending','generating','completed','error') NOT NULL DEFAULT 'pending',
  `error_message` text DEFAULT NULL,
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `challenge_model` (`challenge_id`, `ai_model_id`),
  UNIQUE KEY `challenge_file` (`challenge_id`, `file_number`),
  KEY `idx_challenge` (`challenge_id`),
  KEY `idx_model` (`ai_model_id`),
  CONSTRAINT `fk_output_challenge` FOREIGN KEY (`challenge_id`) REFERENCES `challenges` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_output_model` FOREIGN KEY (`ai_model_id`) REFERENCES `ai_models` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: votes (AI voting AI)
-- --------------------------------------------------------
DROP TABLE IF EXISTS `votes`;
CREATE TABLE `votes` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `challenge_id` int(11) NOT NULL,
  `voter_ai_id` int(11) NOT NULL COMMENT 'AI doing the voting',
  `voted_output_id` int(11) NOT NULL COMMENT 'Output being voted on',
  `functionality` tinyint(3) UNSIGNED NOT NULL DEFAULT 0 COMMENT '1-10',
  `aesthetics` tinyint(3) UNSIGNED NOT NULL DEFAULT 0 COMMENT '1-10',
  `creativity` tinyint(3) UNSIGNED NOT NULL DEFAULT 0 COMMENT '1-10',
  `code_quality` tinyint(3) UNSIGNED NOT NULL DEFAULT 0 COMMENT '1-10',
  `total_score` decimal(4,2) GENERATED ALWAYS AS ((`functionality` + `aesthetics` + `creativity` + `code_quality`) / 4) STORED,
  `feedback` text DEFAULT NULL COMMENT 'AI comment',
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_vote` (`voter_ai_id`, `voted_output_id`),
  KEY `idx_challenge` (`challenge_id`),
  KEY `idx_output` (`voted_output_id`),
  CONSTRAINT `fk_vote_challenge` FOREIGN KEY (`challenge_id`) REFERENCES `challenges` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_vote_voter` FOREIGN KEY (`voter_ai_id`) REFERENCES `ai_models` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_vote_output` FOREIGN KEY (`voted_output_id`) REFERENCES `challenge_outputs` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: user_votes (public voting, future use)
-- --------------------------------------------------------
DROP TABLE IF EXISTS `user_votes`;
CREATE TABLE `user_votes` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `challenge_id` int(11) NOT NULL,
  `voted_output_id` int(11) NOT NULL,
  `session_id` varchar(64) NOT NULL COMMENT 'Prevent duplicate votes',
  `ip_address` varchar(45) DEFAULT NULL,
  `user_agent` varchar(255) DEFAULT NULL,
  `created_at` datetime DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_user_vote` (`challenge_id`, `session_id`),
  KEY `idx_output` (`voted_output_id`),
  CONSTRAINT `fk_uservote_challenge` FOREIGN KEY (`challenge_id`) REFERENCES `challenges` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_uservote_output` FOREIGN KEY (`voted_output_id`) REFERENCES `challenge_outputs` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- --------------------------------------------------------
-- Table: rankings (cached leaderboard)
-- --------------------------------------------------------
DROP TABLE IF EXISTS `rankings`;
CREATE TABLE `rankings` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `ai_model_id` int(11) NOT NULL,
  `total_challenges` int(11) NOT NULL DEFAULT 0,
  `total_wins` int(11) NOT NULL DEFAULT 0,
  `avg_functionality` decimal(4,2) DEFAULT 0.00,
  `avg_aesthetics` decimal(4,2) DEFAULT 0.00,
  `avg_creativity` decimal(4,2) DEFAULT 0.00,
  `avg_code_quality` decimal(4,2) DEFAULT 0.00,
  `avg_total_score` decimal(4,2) DEFAULT 0.00,
  `elo_rating` int(11) NOT NULL DEFAULT 1500,
  `last_updated` datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `ai_model_id` (`ai_model_id`),
  CONSTRAINT `fk_ranking_model` FOREIGN KEY (`ai_model_id`) REFERENCES `ai_models` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Initialize rankings for all models
INSERT INTO `rankings` (`ai_model_id`) 
SELECT `id` FROM `ai_models`;

-- --------------------------------------------------------
-- Table: settings
-- --------------------------------------------------------
DROP TABLE IF EXISTS `settings`;
CREATE TABLE `settings` (
  `setting_key` varchar(50) NOT NULL,
  `setting_value` text DEFAULT NULL,
  `description` varchar(255) DEFAULT NULL,
  PRIMARY KEY (`setting_key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

INSERT INTO `settings` (`setting_key`, `setting_value`, `description`) VALUES
('site_name', 'Betatester.it', 'Site name'),
('api_timeout', '120', 'API timeout in seconds'),
('slug_generator_model', 'claude-sonnet-4-20250514', 'Model used for slug generation'),
('vote_prompt_template', 'You are judging an AI coding challenge. Rate on: functionality, aesthetics, creativity, code_quality (1-10 each). Return JSON only.', 'Voting prompt template'),
('outputs_path', '/challenges/', 'Path where HTML outputs are saved'),
('max_concurrent_requests', '3', 'Max parallel API requests');

SET FOREIGN_KEY_CHECKS = 1;
