<?php
// new-challenge.php v1.0
// Betatester.it - Create New Challenge (Main Page)

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/AIService.php';
requireAuth();

$ai = new AIService();
$models = $ai->getActiveModels(true);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow">
    <title>New Challenge - <?= h(SITE_NAME) ?> Admin</title>
    <link rel="stylesheet" href="assets/admin.css">
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <h1>🤖 Betatester</h1>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php">📊 Dashboard</a>
                <a href="new-challenge.php" class="active">🚀 New Challenge</a>
                <a href="challenges.php">📋 All Challenges</a>
                <a href="ai-models.php">🤖 AI Models</a>
                <a href="settings.php">⚙️ Settings</a>
                <a href="logout.php" class="logout">🚪 Logout</a>
            </nav>
        </aside>
        
        <main class="main-content">
            <header class="page-header">
                <h2>🚀 New Challenge</h2>
            </header>
            
            <div id="alertContainer"></div>
            
            <div class="card" id="formSection">
                <h3>Create a new AI coding challenge</h3>
                
                <form id="challengeForm" onsubmit="return false;">
                    <div class="form-group">
                        <label for="prompt">Challenge Prompt</label>
                        <textarea id="prompt" name="prompt" placeholder="e.g., Create a Snake game in a single HTML file with colorful graphics and smooth animations" required></textarea>
                        <div class="help-text">Be specific about what you want. The AI will generate a single HTML file with everything included.</div>
                    </div>
                    
                    <div class="form-group">
                        <label>Select AI Models</label>
                        <div class="checkbox-grid" id="modelGrid">
                            <?php foreach ($models as $model): ?>
                            <label class="checkbox-item <?= empty($model['api_key']) ? '' : 'checked' ?>">
                                <input type="checkbox" 
                                       name="models[]" 
                                       value="<?= $model['id'] ?>" 
                                       data-name="<?= h($model['name']) ?>"
                                       <?= empty($model['api_key']) ? 'disabled' : 'checked' ?>>
                                <span class="color-dot" style="background: <?= h($model['color_hex']) ?>"></span>
                                <?= h($model['name']) ?>
                                <?php if (empty($model['api_key'])): ?>
                                <span class="no-api">No API key</span>
                                <?php endif; ?>
                            </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <button type="button" class="btn btn-primary" id="startBtn" onclick="startChallenge()">
                        🚀 Start Challenge
                    </button>
                </form>
            </div>
            
            <!-- Progress Section (hidden initially) -->
            <div class="card" id="progressSection" style="display: none;">
                <h3>⏳ Challenge in Progress</h3>
                
                <div id="statusText" class="alert alert-info">
                    Initializing...
                </div>
                
                <div id="progressItems"></div>
            </div>
            
            <!-- Voting Section (hidden initially) -->
            <div class="card" id="votingSection" style="display: none;">
                <h3>🗳️ AI Voting in Progress</h3>
                
                <div id="votingStatus" class="alert alert-info">
                    AIs are evaluating each other's work...
                </div>
                
                <div id="votingProgress"></div>
            </div>
            
            <!-- Results Section (hidden initially) -->
            <div class="card" id="resultsSection" style="display: none;">
                <h3>🏆 Challenge Results</h3>
                
                <div id="resultsGrid" class="results-grid"></div>
                
                <div style="margin-top: 24px; display: flex; gap: 12px;">
                    <a href="#" id="viewChallengeBtn" class="btn btn-primary" target="_blank">View Challenge</a>
                    <a href="new-challenge.php" class="btn btn-secondary">Create Another</a>
                </div>
            </div>
        </main>
    </div>

<script>
// State
let challengeId = null;
let challengeSlug = null;
let fileMapping = [];
let selectedModels = [];

// Get CSRF token (if implementing)
const csrfToken = '';

// Helper: Show alert
function showAlert(message, type = 'info') {
    const container = document.getElementById('alertContainer');
    container.innerHTML = `<div class="alert alert-${type}">${message}</div>`;
}

// Helper: API call
async function apiCall(endpoint, data = {}) {
    const response = await fetch(`api/${endpoint}`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(data)
    });
    return await response.json();
}

// Update checkbox styling
document.querySelectorAll('.checkbox-item input').forEach(cb => {
    cb.addEventListener('change', function() {
        this.closest('.checkbox-item').classList.toggle('checked', this.checked);
    });
});

// Main function: Start Challenge
async function startChallenge() {
    const prompt = document.getElementById('prompt').value.trim();
    if (!prompt) {
        showAlert('Please enter a prompt', 'error');
        return;
    }
    
    // Get selected models
    selectedModels = [];
    document.querySelectorAll('input[name="models[]"]:checked').forEach(cb => {
        selectedModels.push({
            id: parseInt(cb.value),
            name: cb.dataset.name
        });
    });
    
    if (selectedModels.length < 2) {
        showAlert('Please select at least 2 AI models', 'error');
        return;
    }
    
    // Disable form
    document.getElementById('startBtn').disabled = true;
    document.getElementById('formSection').style.opacity = '0.5';
    
    try {
        // Step 1: Generate slug
        showAlert('Generating SEO-friendly slug...', 'info');
        const slugResult = await apiCall('generate-slug.php', { prompt });
        if (!slugResult.success) throw new Error(slugResult.error);
        challengeSlug = slugResult.slug;
        
        // Step 2: Create challenge
        showAlert('Creating challenge...', 'info');
        const createResult = await apiCall('create-challenge.php', {
            prompt,
            slug: challengeSlug,
            model_ids: selectedModels.map(m => m.id)
        });
        if (!createResult.success) throw new Error(createResult.error);
        
        challengeId = createResult.challenge_id;
        fileMapping = createResult.file_mapping;
        
        // Show progress section
        document.getElementById('formSection').style.display = 'none';
        document.getElementById('progressSection').style.display = 'block';
        document.getElementById('alertContainer').innerHTML = '';
        
        // Build progress UI
        buildProgressUI();
        
        // Step 3: Call all AIs in parallel
        await generateFromAllAIs(prompt);
        
        // Step 4: Create files
        updateStatus('Creating HTML files...');
        const filesResult = await apiCall('create-files.php', { challenge_id: challengeId });
        if (!filesResult.success) throw new Error(filesResult.error || 'Failed to create files');
        
        // Step 5: Start voting
        document.getElementById('progressSection').style.display = 'none';
        document.getElementById('votingSection').style.display = 'block';
        await runVoting();
        
        // Step 6: Complete challenge
        const completeResult = await apiCall('complete-challenge.php', { challenge_id: challengeId });
        if (!completeResult.success) throw new Error(completeResult.error);
        
        // Show results
        document.getElementById('votingSection').style.display = 'none';
        showResults(completeResult.standings);
        
    } catch (error) {
        showAlert('Error: ' + error.message, 'error');
        document.getElementById('startBtn').disabled = false;
        document.getElementById('formSection').style.opacity = '1';
    }
}

// Build progress UI
function buildProgressUI() {
    const container = document.getElementById('progressItems');
    container.innerHTML = '';
    
    fileMapping.forEach(item => {
        const model = selectedModels.find(m => m.id === item.model_id);
        const html = `
            <div class="progress-item" id="progress-${item.output_id}">
                <span class="model-name">${model?.name || 'Unknown'}</span>
                <div class="progress-bar">
                    <div class="progress-fill" style="width: 0%"></div>
                </div>
                <span class="progress-status">Waiting...</span>
            </div>
        `;
        container.insertAdjacentHTML('beforeend', html);
    });
}

// Update status text
function updateStatus(text) {
    document.getElementById('statusText').textContent = text;
}

// Update single progress item
function updateProgress(outputId, percent, status, isError = false) {
    const item = document.getElementById(`progress-${outputId}`);
    if (!item) return;
    
    const fill = item.querySelector('.progress-fill');
    const statusEl = item.querySelector('.progress-status');
    
    fill.style.width = percent + '%';
    statusEl.textContent = status;
    
    if (percent === 100) {
        item.classList.add('completed');
    }
    if (isError) {
        item.classList.add('error');
    }
}

// Generate from all AIs in parallel
async function generateFromAllAIs(prompt) {
    updateStatus('Generating code from AI models...');
    
    const promises = fileMapping.map(async (item) => {
        updateProgress(item.output_id, 10, 'Starting...');
        
        try {
            updateProgress(item.output_id, 30, 'Calling API...');
            
            const result = await apiCall('call-ai.php', {
                output_id: item.output_id,
                challenge_id: challengeId,
                model_id: item.model_id
            });
            
            if (result.success) {
                updateProgress(item.output_id, 100, `Done (${(result.time_ms/1000).toFixed(1)}s)`);
                return { ...item, success: true, result };
            } else {
                updateProgress(item.output_id, 100, 'Error', true);
                return { ...item, success: false, error: result.error };
            }
        } catch (e) {
            updateProgress(item.output_id, 100, 'Failed', true);
            return { ...item, success: false, error: e.message };
        }
    });
    
    const results = await Promise.all(promises);
    const successCount = results.filter(r => r.success).length;
    updateStatus(`Generation complete: ${successCount}/${results.length} successful`);
    
    return results;
}

// Run voting process
async function runVoting() {
    document.getElementById('votingStatus').textContent = 'AIs are evaluating each other\'s work...';
    
    // Get all outputs that need voting
    const outputs = fileMapping.filter(f => f.output_id);
    const voters = selectedModels.filter(m => true); // All models can vote
    
    // Build voting progress UI
    const container = document.getElementById('votingProgress');
    container.innerHTML = '';
    
    let totalVotes = voters.length * outputs.length;
    let completedVotes = 0;
    
    // Create progress bar for voting
    container.innerHTML = `
        <div class="progress-item">
            <span class="model-name">Voting</span>
            <div class="progress-bar">
                <div class="progress-fill" id="votingFill" style="width: 0%"></div>
            </div>
            <span class="progress-status" id="votingCount">0/${totalVotes}</span>
        </div>
    `;
    
    // Run all votes (can be parallelized with some limit)
    for (const voter of voters) {
        for (const output of outputs) {
            try {
                await apiCall('vote.php', {
                    challenge_id: challengeId,
                    voter_model_id: voter.id,
                    output_id: output.output_id
                });
            } catch (e) {
                console.error('Vote error:', e);
            }
            
            completedVotes++;
            const percent = Math.round((completedVotes / totalVotes) * 100);
            document.getElementById('votingFill').style.width = percent + '%';
            document.getElementById('votingCount').textContent = `${completedVotes}/${totalVotes}`;
        }
    }
    
    document.getElementById('votingStatus').textContent = 'Voting complete! Calculating results...';
}

// Show final results
function showResults(standings) {
    document.getElementById('resultsSection').style.display = 'block';
    
    const container = document.getElementById('resultsGrid');
    container.innerHTML = '';
    
    standings.forEach((standing, index) => {
        const isWinner = standing.is_winner;
        const html = `
            <div class="result-card ${isWinner ? 'winner' : ''}">
                <div class="rank">#${standing.rank}</div>
                <div class="model-name">
                    <span class="color-dot" style="background: ${standing.color}"></span>
                    ${standing.model_name}
                    ${isWinner ? '🏆' : ''}
                </div>
                <div class="score">${standing.scores.total.toFixed(2)}</div>
                <div class="score-breakdown">
                    <span>Functionality: ${standing.scores.functionality}</span>
                    <span>Aesthetics: ${standing.scores.aesthetics}</span>
                    <span>Creativity: ${standing.scores.creativity}</span>
                    <span>Code Quality: ${standing.scores.code_quality}</span>
                </div>
            </div>
        `;
        container.insertAdjacentHTML('beforeend', html);
    });
    
    // Set view link
    document.getElementById('viewChallengeBtn').href = `<?= CHALLENGES_URL ?>/${challengeSlug}/`;
}
</script>
</body>
</html>
