<?php
// config.php v1.0
// Betatester.it - Configuration

// Prevent direct access
if (basename($_SERVER['PHP_SELF']) === 'config.php') {
    die('Direct access not allowed');
}

// ============================================
// DATABASE CONFIGURATION
// ============================================
define('DB_HOST', 'localhost');
define('DB_NAME', 'beta59221cf_db');
define('DB_USER', 'beta59221cf_utente');
define('DB_PASS', 'qTe!_lHwYk5Z');
define('DB_CHARSET', 'utf8mb4');

// ============================================
// SITE CONFIGURATION
// ============================================
define('SITE_NAME', 'Betatester.it');
define('SITE_URL', 'https://www.betatester.it');
define('ADMIN_PATH', '/admin');

// ============================================
// API CONFIGURATION
// ============================================
define('API_TIMEOUT', 120); // seconds
define('MAX_CONCURRENT_REQUESTS', 3);

// ============================================
// PATHS
// ============================================
define('ROOT_PATH', dirname(__DIR__));
define('ADMIN_DIR', __DIR__);
define('CHALLENGES_PATH', ROOT_PATH . '/challenges');
define('CHALLENGES_URL', '/challenges');

// ============================================
// SESSION CONFIGURATION
// ============================================
define('SESSION_NAME', 'betatester_admin');
define('SESSION_LIFETIME', 86400); // 24 hours

// ============================================
// DATABASE CONNECTION (PDO)
// ============================================
function getDB(): PDO {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=' . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES " . DB_CHARSET
            ];
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            error_log('Database connection failed: ' . $e->getMessage());
            die('Database connection error. Please try again later.');
        }
    }
    
    return $pdo;
}

// ============================================
// HELPER FUNCTIONS
// ============================================

/**
 * Sanitize output for HTML
 */
function h(string $str): string {
    return htmlspecialchars($str, ENT_QUOTES, 'UTF-8');
}

/**
 * JSON response helper
 */
function jsonResponse(array $data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Get setting from database
 */
function getSetting(string $key, string $default = ''): string {
    static $cache = [];
    
    if (!isset($cache[$key])) {
        $db = getDB();
        $stmt = $db->prepare('SELECT setting_value FROM settings WHERE setting_key = ?');
        $stmt->execute([$key]);
        $result = $stmt->fetchColumn();
        $cache[$key] = $result !== false ? $result : $default;
    }
    
    return $cache[$key];
}

/**
 * Update setting in database
 */
function setSetting(string $key, string $value): bool {
    $db = getDB();
    $stmt = $db->prepare('INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) 
                          ON DUPLICATE KEY UPDATE setting_value = ?');
    return $stmt->execute([$key, $value, $value]);
}

/**
 * Generate CSRF token
 */
function generateCSRF(): string {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 */
function verifyCSRF(string $token): bool {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Log activity
 */
function logActivity(string $action, ?array $data = null): void {
    $log = date('Y-m-d H:i:s') . ' | ' . $action;
    if ($data) {
        $log .= ' | ' . json_encode($data);
    }
    error_log($log, 3, ROOT_PATH . '/logs/activity.log');
}

// ============================================
// API PROVIDER CONFIGURATIONS
// ============================================
function getAPIConfig(string $provider): array {
    $configs = [
        'anthropic' => [
            'header_auth' => 'x-api-key',
            'header_version' => 'anthropic-version: 2023-06-01',
            'content_path' => 'content.0.text',
            'request_format' => 'anthropic'
        ],
        'openai' => [
            'header_auth' => 'Authorization: Bearer',
            'content_path' => 'choices.0.message.content',
            'request_format' => 'openai'
        ],
        'google' => [
            'header_auth' => 'x-goog-api-key',
            'content_path' => 'candidates.0.content.parts.0.text',
            'request_format' => 'google'
        ],
        'xai' => [
            'header_auth' => 'Authorization: Bearer',
            'content_path' => 'choices.0.message.content',
            'request_format' => 'openai' // Grok uses OpenAI format
        ],
        'deepseek' => [
            'header_auth' => 'Authorization: Bearer',
            'content_path' => 'choices.0.message.content',
            'request_format' => 'openai' // DeepSeek uses OpenAI format
        ]
    ];
    
    return $configs[$provider] ?? $configs['openai'];
}
