<?php
// api/vote.php v1.0
// Betatester.it - Single AI votes on single output

require_once dirname(__DIR__) . '/auth.php';
require_once dirname(__DIR__) . '/AIService.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'error' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'error' => 'Method not allowed'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);
$challengeId = (int) ($input['challenge_id'] ?? 0);
$voterModelId = (int) ($input['voter_model_id'] ?? 0);
$outputId = (int) ($input['output_id'] ?? 0);

if (!$challengeId || !$voterModelId || !$outputId) {
    jsonResponse(['success' => false, 'error' => 'Missing required parameters']);
}

$db = getDB();
$ai = new AIService();

// Get voter model
$voterModel = $ai->getModel($voterModelId);
if (!$voterModel || !$voterModel['can_vote']) {
    jsonResponse(['success' => false, 'error' => 'Voter model not found or cannot vote']);
}

// Get challenge
$stmt = $db->prepare('SELECT prompt_text FROM challenges WHERE id = ?');
$stmt->execute([$challengeId]);
$challenge = $stmt->fetch();

if (!$challenge) {
    jsonResponse(['success' => false, 'error' => 'Challenge not found']);
}

// Get output to vote on
$stmt = $db->prepare('SELECT output_html FROM challenge_outputs WHERE id = ? AND challenge_id = ?');
$stmt->execute([$outputId, $challengeId]);
$output = $stmt->fetch();

if (!$output || empty($output['output_html'])) {
    jsonResponse(['success' => false, 'error' => 'Output not found or empty']);
}

// Check if vote already exists
$stmt = $db->prepare('SELECT id FROM votes WHERE voter_ai_id = ? AND voted_output_id = ?');
$stmt->execute([$voterModelId, $outputId]);
if ($stmt->fetch()) {
    jsonResponse(['success' => false, 'error' => 'Vote already exists']);
}

// Get vote from AI
$result = $ai->getVote($voterModel, $challenge['prompt_text'], $output['output_html']);

if (!$result['success']) {
    jsonResponse([
        'success' => false,
        'error' => $result['error'] ?? 'Failed to get vote',
        'raw' => $result['raw'] ?? null
    ]);
}

$votes = $result['votes'];

// Save vote to database
$stmt = $db->prepare('INSERT INTO votes (challenge_id, voter_ai_id, voted_output_id, functionality, aesthetics, creativity, code_quality, feedback) VALUES (?, ?, ?, ?, ?, ?, ?, ?)');

$stmt->execute([
    $challengeId,
    $voterModelId,
    $outputId,
    $votes['functionality'],
    $votes['aesthetics'],
    $votes['creativity'],
    $votes['code_quality'],
    $votes['feedback'] ?? null
]);

$voteId = $db->lastInsertId();

// Calculate total
$total = ($votes['functionality'] + $votes['aesthetics'] + $votes['creativity'] + $votes['code_quality']) / 4;

jsonResponse([
    'success' => true,
    'vote_id' => (int) $voteId,
    'voter' => $voterModel['name'],
    'scores' => $votes,
    'total_score' => round($total, 2),
    'time_ms' => $result['time_ms']
]);
