<?php
// api/create-files.php v1.0
// Betatester.it - Create physical HTML files from database

require_once dirname(__DIR__) . '/auth.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'error' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'error' => 'Method not allowed'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);
$challengeId = (int) ($input['challenge_id'] ?? 0);

if (!$challengeId) {
    jsonResponse(['success' => false, 'error' => 'Challenge ID is required']);
}

$db = getDB();

// Get challenge
$stmt = $db->prepare('SELECT * FROM challenges WHERE id = ?');
$stmt->execute([$challengeId]);
$challenge = $stmt->fetch();

if (!$challenge) {
    jsonResponse(['success' => false, 'error' => 'Challenge not found']);
}

// Get all completed outputs
$stmt = $db->prepare('SELECT * FROM challenge_outputs WHERE challenge_id = ? AND status = ?');
$stmt->execute([$challengeId, 'completed']);
$outputs = $stmt->fetchAll();

if (empty($outputs)) {
    jsonResponse(['success' => false, 'error' => 'No completed outputs found']);
}

// Create directory
$challengeDir = CHALLENGES_PATH . '/' . $challenge['slug'];

if (!is_dir($challengeDir)) {
    if (!mkdir($challengeDir, 0755, true)) {
        jsonResponse(['success' => false, 'error' => 'Failed to create directory']);
    }
}

$filesCreated = [];
$errors = [];

foreach ($outputs as $output) {
    $filePath = $challengeDir . '/' . $output['file_number'] . '.html';
    
    if (file_put_contents($filePath, $output['output_html']) !== false) {
        $filesCreated[] = [
            'file_number' => $output['file_number'],
            'path' => $output['file_path'],
            'size' => strlen($output['output_html'])
        ];
    } else {
        $errors[] = 'Failed to write file ' . $output['file_number'] . '.html';
    }
}

// Create index.html with iframe selector for the challenge
$indexContent = generateChallengeIndex($challenge, $outputs);
$indexPath = $challengeDir . '/index.html';

if (file_put_contents($indexPath, $indexContent) === false) {
    $errors[] = 'Failed to write index.html';
}

jsonResponse([
    'success' => empty($errors),
    'files_created' => $filesCreated,
    'directory' => $challenge['slug'],
    'errors' => $errors
]);

/**
 * Generate index.html for challenge viewing
 */
function generateChallengeIndex(array $challenge, array $outputs): string {
    $totalOutputs = count($outputs);
    
    $optionsHtml = '';
    for ($i = 1; $i <= $totalOutputs; $i++) {
        $optionsHtml .= "<option value=\"{$i}.html\">Version {$i}</option>\n";
    }
    
    return '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="google-site-verification" content="y2x30EDuOSuwvftVqNIjIwo5PGq4Kyv8VphNPTyKymE" />
    <title>' . htmlspecialchars($challenge['slug']) . ' - AI Challenge | Betatester.it</title>
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            background: #0f0f1a;
            color: #fff;
            min-height: 100vh;
        }
        .header {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            gap: 15px;
            border-bottom: 1px solid #333;
        }
        .logo {
            font-size: 24px;
            font-weight: bold;
            color: #4a6cf7;
            text-decoration: none;
        }
        .controls {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        select {
            padding: 10px 20px;
            border-radius: 8px;
            border: 2px solid #333;
            background: #1a1a2e;
            color: #fff;
            font-size: 16px;
            cursor: pointer;
        }
        select:hover { border-color: #4a6cf7; }
        .btn {
            padding: 10px 20px;
            border-radius: 8px;
            border: none;
            background: #4a6cf7;
            color: #fff;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover { background: #3b5ce4; }
        .btn-outline {
            background: transparent;
            border: 2px solid #4a6cf7;
        }
        .iframe-container {
            width: 100%;
            height: calc(100vh - 80px);
        }
        iframe {
            width: 100%;
            height: 100%;
            border: none;
            background: #fff;
        }
        .prompt-bar {
            background: #1a1a2e;
            padding: 10px 20px;
            font-size: 14px;
            color: #888;
            border-bottom: 1px solid #333;
        }
        .prompt-bar strong { color: #4a6cf7; }
    </style>
</head>
<body>
    <div class="header">
        <a href="/" class="logo">🤖 Betatester.it</a>
        <div class="controls">
            <select id="versionSelect" onchange="changeVersion()">
                ' . $optionsHtml . '
            </select>
            <a href="?reveal=1" class="btn btn-outline">Reveal AI</a>
            <a href="/" class="btn">Back to Home</a>
        </div>
    </div>
    <div class="prompt-bar">
        <strong>Challenge:</strong> ' . htmlspecialchars($challenge['prompt_text']) . '
    </div>
    <div class="iframe-container">
        <iframe id="outputFrame" src="1.html" sandbox="allow-scripts allow-same-origin"></iframe>
    </div>
    <script>
        function changeVersion() {
            const select = document.getElementById("versionSelect");
            const iframe = document.getElementById("outputFrame");
            iframe.src = select.value;
        }
        
        // Check for reveal parameter
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.get("reveal") === "1") {
            // Fetch and show AI names (would need AJAX call to backend)
            console.log("Reveal mode enabled");
        }
    </script>
</body>
</html>';
}
