<?php
// api/complete-challenge.php v1.0
// Betatester.it - Mark challenge as completed and update rankings

require_once dirname(__DIR__) . '/auth.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'error' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'error' => 'Method not allowed'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);
$challengeId = (int) ($input['challenge_id'] ?? 0);

if (!$challengeId) {
    jsonResponse(['success' => false, 'error' => 'Challenge ID is required']);
}

$db = getDB();

try {
    $db->beginTransaction();
    
    // Update challenge status
    $stmt = $db->prepare('UPDATE challenges SET status = ?, completed_at = NOW(), is_published = 1 WHERE id = ?');
    $stmt->execute(['completed', $challengeId]);
    
    // Calculate results for this challenge
    $stmt = $db->prepare('
        SELECT 
            co.ai_model_id,
            co.id as output_id,
            AVG(v.functionality) as avg_functionality,
            AVG(v.aesthetics) as avg_aesthetics,
            AVG(v.creativity) as avg_creativity,
            AVG(v.code_quality) as avg_code_quality,
            AVG(v.total_score) as avg_total
        FROM challenge_outputs co
        LEFT JOIN votes v ON v.voted_output_id = co.id
        WHERE co.challenge_id = ?
        GROUP BY co.ai_model_id, co.id
        ORDER BY avg_total DESC
    ');
    $stmt->execute([$challengeId]);
    $results = $stmt->fetchAll();
    
    // Determine winner
    $winnerId = null;
    $winnerScore = 0;
    foreach ($results as $result) {
        if ($result['avg_total'] > $winnerScore) {
            $winnerScore = $result['avg_total'];
            $winnerId = $result['ai_model_id'];
        }
    }
    
    // Update rankings for each AI
    foreach ($results as $result) {
        $isWinner = ($result['ai_model_id'] == $winnerId) ? 1 : 0;
        
        $stmt = $db->prepare('
            UPDATE rankings SET
                total_challenges = total_challenges + 1,
                total_wins = total_wins + ?,
                avg_functionality = (avg_functionality * (total_challenges - 1) + ?) / total_challenges,
                avg_aesthetics = (avg_aesthetics * (total_challenges - 1) + ?) / total_challenges,
                avg_creativity = (avg_creativity * (total_challenges - 1) + ?) / total_challenges,
                avg_code_quality = (avg_code_quality * (total_challenges - 1) + ?) / total_challenges,
                avg_total_score = (avg_total_score * (total_challenges - 1) + ?) / total_challenges,
                last_updated = NOW()
            WHERE ai_model_id = ?
        ');
        
        $stmt->execute([
            $isWinner,
            $result['avg_functionality'] ?? 0,
            $result['avg_aesthetics'] ?? 0,
            $result['avg_creativity'] ?? 0,
            $result['avg_code_quality'] ?? 0,
            $result['avg_total'] ?? 0,
            $result['ai_model_id']
        ]);
    }
    
    $db->commit();
    
    // Get final standings
    $standings = [];
    foreach ($results as $index => $result) {
        $stmt = $db->prepare('SELECT name, color_hex FROM ai_models WHERE id = ?');
        $stmt->execute([$result['ai_model_id']]);
        $model = $stmt->fetch();
        
        $standings[] = [
            'rank' => $index + 1,
            'model_id' => $result['ai_model_id'],
            'model_name' => $model['name'],
            'color' => $model['color_hex'],
            'scores' => [
                'functionality' => round($result['avg_functionality'], 2),
                'aesthetics' => round($result['avg_aesthetics'], 2),
                'creativity' => round($result['avg_creativity'], 2),
                'code_quality' => round($result['avg_code_quality'], 2),
                'total' => round($result['avg_total'], 2)
            ],
            'is_winner' => $result['ai_model_id'] == $winnerId
        ];
    }
    
    jsonResponse([
        'success' => true,
        'challenge_id' => $challengeId,
        'winner_id' => $winnerId,
        'standings' => $standings
    ]);
    
} catch (Exception $e) {
    $db->rollBack();
    jsonResponse(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
}
