<?php
// api/call-ai.php v1.0
// Betatester.it - Call single AI model to generate code

require_once dirname(__DIR__) . '/auth.php';
require_once dirname(__DIR__) . '/AIService.php';

header('Content-Type: application/json');

if (!isLoggedIn()) {
    jsonResponse(['success' => false, 'error' => 'Unauthorized'], 401);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['success' => false, 'error' => 'Method not allowed'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);
$outputId = (int) ($input['output_id'] ?? 0);
$challengeId = (int) ($input['challenge_id'] ?? 0);
$modelId = (int) ($input['model_id'] ?? 0);

if (!$outputId || !$challengeId || !$modelId) {
    jsonResponse(['success' => false, 'error' => 'Missing required parameters']);
}

$db = getDB();
$ai = new AIService();

// Get model
$model = $ai->getModel($modelId);
if (!$model) {
    jsonResponse(['success' => false, 'error' => 'AI model not found or inactive']);
}

// Get challenge prompt
$stmt = $db->prepare('SELECT prompt_text, slug FROM challenges WHERE id = ?');
$stmt->execute([$challengeId]);
$challenge = $stmt->fetch();

if (!$challenge) {
    jsonResponse(['success' => false, 'error' => 'Challenge not found']);
}

// Update status to generating
$stmt = $db->prepare('UPDATE challenge_outputs SET status = ? WHERE id = ?');
$stmt->execute(['generating', $outputId]);

// Build the code generation prompt
$codePrompt = 'Create a complete, working single-file HTML page for the following request:

"' . $challenge['prompt_text'] . '"

Requirements:
- Everything must be in ONE HTML file (HTML, CSS, JavaScript together)
- Use modern, clean design with nice colors and animations
- Make it fully functional and interactive
- Include proper viewport meta tag for mobile
- Use only vanilla JavaScript (no external libraries unless absolutely necessary)
- Add smooth animations and transitions where appropriate
- Make it visually appealing with gradients, shadows, and modern styling

Return ONLY the complete HTML code, starting with <!DOCTYPE html> and ending with </html>.
No explanations, no markdown code blocks, just the raw HTML.';

$result = $ai->callAI($model, $codePrompt);

if (!$result['success']) {
    $stmt = $db->prepare('UPDATE challenge_outputs SET status = ?, error_message = ? WHERE id = ?');
    $stmt->execute(['error', $result['error'], $outputId]);
    jsonResponse(['success' => false, 'error' => $result['error']]);
}

// Clean up the response - extract HTML
$html = $result['content'];

// Remove markdown code blocks if present
$html = preg_replace('/^```html?\s*/i', '', $html);
$html = preg_replace('/\s*```$/', '', $html);
$html = trim($html);

// Ensure it starts with DOCTYPE or html tag
if (stripos($html, '<!DOCTYPE') !== 0 && stripos($html, '<html') !== 0) {
    // Try to find the start of HTML
    if (preg_match('/<!DOCTYPE html>.*$/is', $html, $matches)) {
        $html = $matches[0];
    }
}

// Get file number for this output
$stmt = $db->prepare('SELECT file_number FROM challenge_outputs WHERE id = ?');
$stmt->execute([$outputId]);
$output = $stmt->fetch();
$fileNumber = $output['file_number'];

// Build file path
$filePath = '/' . $challenge['slug'] . '/' . $fileNumber . '.html';

// Save to database
$stmt = $db->prepare('UPDATE challenge_outputs SET output_html = ?, file_path = ?, generation_time_ms = ?, status = ? WHERE id = ?');
$stmt->execute([$html, $filePath, $result['time_ms'], 'completed', $outputId]);

jsonResponse([
    'success' => true,
    'output_id' => $outputId,
    'model_name' => $model['name'],
    'file_number' => $fileNumber,
    'file_path' => $filePath,
    'time_ms' => $result['time_ms'],
    'html_length' => strlen($html)
]);
