<?php
// ai-models.php v1.0
// Betatester.it - Manage AI Models

require_once __DIR__ . '/auth.php';
requireAuth();

$db = getDB();
$message = '';
$messageType = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'update') {
        $id = (int) $_POST['id'];
        $apiKey = trim($_POST['api_key'] ?? '');
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        $canGenerate = isset($_POST['can_generate']) ? 1 : 0;
        $canVote = isset($_POST['can_vote']) ? 1 : 0;
        
        $stmt = $db->prepare('UPDATE ai_models SET api_key = ?, is_active = ?, can_generate = ?, can_vote = ?, updated_at = NOW() WHERE id = ?');
        $stmt->execute([$apiKey, $isActive, $canGenerate, $canVote, $id]);
        
        $message = 'Model updated successfully';
        $messageType = 'success';
    }
    
    if ($action === 'add') {
        $name = trim($_POST['name'] ?? '');
        $provider = trim($_POST['provider'] ?? '');
        $modelCode = trim($_POST['model_code'] ?? '');
        $apiEndpoint = trim($_POST['api_endpoint'] ?? '');
        $apiKey = trim($_POST['api_key'] ?? '');
        $colorHex = trim($_POST['color_hex'] ?? '#666666');
        
        if ($name && $provider && $modelCode) {
            $stmt = $db->prepare('INSERT INTO ai_models (name, provider, model_code, api_endpoint, api_key, color_hex) VALUES (?, ?, ?, ?, ?, ?)');
            $stmt->execute([$name, $provider, $modelCode, $apiEndpoint, $apiKey, $colorHex]);
            
            // Add to rankings
            $modelId = $db->lastInsertId();
            $db->prepare('INSERT INTO rankings (ai_model_id) VALUES (?)')->execute([$modelId]);
            
            $message = 'Model added successfully';
            $messageType = 'success';
        } else {
            $message = 'Please fill all required fields';
            $messageType = 'error';
        }
    }
}

// Get all models
$models = $db->query('SELECT * FROM ai_models ORDER BY display_order ASC, id ASC')->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow">
    <title>AI Models - <?= h(SITE_NAME) ?> Admin</title>
    <link rel="stylesheet" href="assets/admin.css">
    <style>
        .model-card {
            background: var(--bg-input);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 16px;
        }
        .model-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }
        .model-name {
            font-size: 18px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .model-provider {
            font-size: 13px;
            color: var(--text-muted);
            background: var(--bg-card);
            padding: 4px 10px;
            border-radius: 4px;
        }
        .model-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 16px;
        }
        .api-key-input {
            font-family: monospace;
            font-size: 13px;
        }
        .toggle-group {
            display: flex;
            gap: 20px;
            margin-top: 16px;
        }
        .toggle-group label {
            display: flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
        }
        .toggle-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
        }
        .model-status {
            display: flex;
            gap: 8px;
        }
        .status-badge {
            font-size: 11px;
            padding: 3px 8px;
            border-radius: 4px;
        }
        .status-badge.active { background: rgba(16, 185, 129, 0.2); color: var(--success); }
        .status-badge.inactive { background: rgba(239, 68, 68, 0.2); color: var(--danger); }
        .status-badge.has-key { background: rgba(59, 130, 246, 0.2); color: var(--info); }
        .status-badge.no-key { background: rgba(245, 158, 11, 0.2); color: var(--warning); }
        
        .add-model-form {
            display: none;
            margin-top: 20px;
        }
        .add-model-form.active { display: block; }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <h1>🤖 Betatester</h1>
                <span>Admin Panel</span>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php">📊 Dashboard</a>
                <a href="new-challenge.php">🚀 New Challenge</a>
                <a href="challenges.php">📋 All Challenges</a>
                <a href="ai-models.php" class="active">🤖 AI Models</a>
                <a href="settings.php">⚙️ Settings</a>
                <a href="logout.php" class="logout">🚪 Logout</a>
            </nav>
        </aside>
        
        <main class="main-content">
            <header class="page-header">
                <h2>🤖 AI Models</h2>
                <button class="btn btn-primary" onclick="toggleAddForm()">+ Add Model</button>
            </header>
            
            <?php if ($message): ?>
            <div class="alert alert-<?= $messageType ?>"><?= h($message) ?></div>
            <?php endif; ?>
            
            <!-- Add Model Form -->
            <div class="card add-model-form" id="addModelForm">
                <h3>Add New AI Model</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="add">
                    <div class="model-grid">
                        <div class="form-group">
                            <label>Name *</label>
                            <input type="text" name="name" placeholder="Claude Sonnet 4" required>
                        </div>
                        <div class="form-group">
                            <label>Provider *</label>
                            <select name="provider" required>
                                <option value="anthropic">Anthropic</option>
                                <option value="openai">OpenAI</option>
                                <option value="google">Google</option>
                                <option value="xai">xAI</option>
                                <option value="deepseek">DeepSeek</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Model Code *</label>
                            <input type="text" name="model_code" placeholder="claude-sonnet-4-20250514" required>
                        </div>
                        <div class="form-group">
                            <label>API Endpoint</label>
                            <input type="text" name="api_endpoint" placeholder="https://api.anthropic.com/v1/messages">
                        </div>
                        <div class="form-group">
                            <label>API Key</label>
                            <input type="password" name="api_key" placeholder="sk-...">
                        </div>
                        <div class="form-group">
                            <label>Brand Color</label>
                            <input type="color" name="color_hex" value="#666666">
                        </div>
                    </div>
                    <button type="submit" class="btn btn-success">Add Model</button>
                    <button type="button" class="btn btn-secondary" onclick="toggleAddForm()">Cancel</button>
                </form>
            </div>
            
            <!-- Existing Models -->
            <?php foreach ($models as $model): ?>
            <div class="model-card">
                <form method="POST">
                    <input type="hidden" name="action" value="update">
                    <input type="hidden" name="id" value="<?= $model['id'] ?>">
                    
                    <div class="model-header">
                        <div class="model-name">
                            <span class="color-dot" style="background: <?= h($model['color_hex']) ?>"></span>
                            <?= h($model['name']) ?>
                            <span class="model-provider"><?= h($model['provider']) ?></span>
                        </div>
                        <div class="model-status">
                            <span class="status-badge <?= $model['is_active'] ? 'active' : 'inactive' ?>">
                                <?= $model['is_active'] ? 'Active' : 'Inactive' ?>
                            </span>
                            <span class="status-badge <?= !empty($model['api_key']) ? 'has-key' : 'no-key' ?>">
                                <?= !empty($model['api_key']) ? 'API Key Set' : 'No API Key' ?>
                            </span>
                        </div>
                    </div>
                    
                    <div class="model-grid">
                        <div class="form-group">
                            <label>Model Code</label>
                            <input type="text" value="<?= h($model['model_code']) ?>" disabled>
                        </div>
                        <div class="form-group">
                            <label>API Key</label>
                            <input type="password" name="api_key" value="<?= h($model['api_key']) ?>" 
                                   class="api-key-input" placeholder="Enter API key">
                        </div>
                    </div>
                    
                    <div class="toggle-group">
                        <label>
                            <input type="checkbox" name="is_active" <?= $model['is_active'] ? 'checked' : '' ?>>
                            Active
                        </label>
                        <label>
                            <input type="checkbox" name="can_generate" <?= $model['can_generate'] ? 'checked' : '' ?>>
                            Can Generate
                        </label>
                        <label>
                            <input type="checkbox" name="can_vote" <?= $model['can_vote'] ? 'checked' : '' ?>>
                            Can Vote
                        </label>
                        <button type="submit" class="btn btn-secondary" style="margin-left: auto;">Save Changes</button>
                    </div>
                </form>
            </div>
            <?php endforeach; ?>
        </main>
    </div>
    
    <script>
    function toggleAddForm() {
        document.getElementById('addModelForm').classList.toggle('active');
    }
    </script>
</body>
</html>
