<?php
// AIService.php v1.0
// Betatester.it - AI API Service Handler

require_once __DIR__ . '/config.php';

class AIService {
    
    private PDO $db;
    private int $timeout;
    
    public function __construct() {
        $this->db = getDB();
        $this->timeout = (int) getSetting('api_timeout', '120');
    }
    
    /**
     * Get AI model by ID
     */
    public function getModel(int $id): ?array {
        $stmt = $this->db->prepare('SELECT * FROM ai_models WHERE id = ? AND is_active = 1');
        $stmt->execute([$id]);
        return $stmt->fetch() ?: null;
    }
    
    /**
     * Get all active models
     */
    public function getActiveModels(bool $canGenerate = true): array {
        $sql = 'SELECT * FROM ai_models WHERE is_active = 1';
        if ($canGenerate) {
            $sql .= ' AND can_generate = 1';
        }
        $sql .= ' ORDER BY display_order ASC';
        
        return $this->db->query($sql)->fetchAll();
    }
    
    /**
     * Call AI API with prompt
     */
    public function callAI(array $model, string $prompt, string $systemPrompt = ''): array {
        $startTime = microtime(true);
        
        if (empty($model['api_key'])) {
            return [
                'success' => false,
                'error' => 'API key not configured for ' . $model['name'],
                'time_ms' => 0
            ];
        }
        
        $config = getAPIConfig($model['provider']);
        
        try {
            $response = $this->makeRequest($model, $prompt, $systemPrompt, $config);
            $timeMs = (int) ((microtime(true) - $startTime) * 1000);
            
            if (isset($response['error'])) {
                return [
                    'success' => false,
                    'error' => $response['error'],
                    'time_ms' => $timeMs
                ];
            }
            
            $content = $this->extractContent($response, $config['content_path']);
            
            return [
                'success' => true,
                'content' => $content,
                'time_ms' => $timeMs,
                'raw_response' => $response
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'time_ms' => (int) ((microtime(true) - $startTime) * 1000)
            ];
        }
    }
    
    /**
     * Make HTTP request to AI API
     */
    private function makeRequest(array $model, string $prompt, string $systemPrompt, array $config): array {
        $ch = curl_init();
        
        $headers = ['Content-Type: application/json'];
        $requestBody = $this->buildRequestBody($model, $prompt, $systemPrompt, $config);
        $url = $model['api_endpoint'];
        
        // Provider-specific configurations
        switch ($model['provider']) {
            case 'anthropic':
                $headers[] = 'x-api-key: ' . $model['api_key'];
                $headers[] = 'anthropic-version: 2023-06-01';
                break;
                
            case 'google':
                $url = $model['api_endpoint'] . $model['model_code'] . ':generateContent?key=' . $model['api_key'];
                break;
                
            case 'openai':
            case 'xai':
            case 'deepseek':
            default:
                $headers[] = 'Authorization: Bearer ' . $model['api_key'];
                break;
        }
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($requestBody),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_CONNECTTIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['error' => 'cURL error: ' . $error];
        }
        
        $decoded = json_decode($response, true);
        
        if ($httpCode !== 200) {
            $errorMsg = $decoded['error']['message'] ?? $decoded['error'] ?? 'HTTP ' . $httpCode;
            return ['error' => $errorMsg];
        }
        
        return $decoded;
    }
    
    /**
     * Build request body based on provider format
     */
    private function buildRequestBody(array $model, string $prompt, string $systemPrompt, array $config): array {
        switch ($config['request_format']) {
            case 'anthropic':
                $body = [
                    'model' => $model['model_code'],
                    'max_tokens' => 8192,
                    'messages' => [
                        ['role' => 'user', 'content' => $prompt]
                    ]
                ];
                if ($systemPrompt) {
                    $body['system'] = $systemPrompt;
                }
                return $body;
                
            case 'google':
                $body = [
                    'contents' => [
                        ['parts' => [['text' => $prompt]]]
                    ],
                    'generationConfig' => [
                        'maxOutputTokens' => 8192
                    ]
                ];
                if ($systemPrompt) {
                    $body['systemInstruction'] = ['parts' => [['text' => $systemPrompt]]];
                }
                return $body;
                
            case 'openai':
            default:
                $messages = [];
                if ($systemPrompt) {
                    $messages[] = ['role' => 'system', 'content' => $systemPrompt];
                }
                $messages[] = ['role' => 'user', 'content' => $prompt];
                
                return [
                    'model' => $model['model_code'],
                    'max_tokens' => 8192,
                    'messages' => $messages
                ];
        }
    }
    
    /**
     * Extract content from nested response using dot notation path
     */
    private function extractContent(array $response, string $path): string {
        $keys = explode('.', $path);
        $value = $response;
        
        foreach ($keys as $key) {
            if (is_array($value) && isset($value[$key])) {
                $value = $value[$key];
            } else {
                return '';
            }
        }
        
        return is_string($value) ? $value : '';
    }
    
    /**
     * Generate SEO slug using Claude
     */
    public function generateSlug(string $prompt): array {
        // Get Claude model for slug generation
        $stmt = $this->db->prepare("SELECT * FROM ai_models WHERE provider = 'anthropic' AND is_active = 1 LIMIT 1");
        $stmt->execute();
        $claude = $stmt->fetch();
        
        if (!$claude) {
            // Fallback: generate simple slug
            return [
                'success' => true,
                'slug' => $this->simpleSlug($prompt)
            ];
        }
        
        $slugPrompt = 'Given this prompt for an AI coding challenge:
"' . $prompt . '"

Generate a SEO-friendly URL slug in English.
Rules:
- Max 50 characters
- Lowercase, hyphens only, no underscores
- Descriptive but concise
- No stop words (the, a, an, etc.)

Return ONLY the slug, nothing else. No quotes, no explanation.';
        
        $result = $this->callAI($claude, $slugPrompt);
        
        if ($result['success']) {
            $slug = strtolower(trim($result['content']));
            $slug = preg_replace('/[^a-z0-9-]/', '', $slug);
            $slug = preg_replace('/-+/', '-', $slug);
            $slug = trim($slug, '-');
            
            if (strlen($slug) > 50) {
                $slug = substr($slug, 0, 50);
                $slug = preg_replace('/-[^-]*$/', '', $slug);
            }
            
            return ['success' => true, 'slug' => $slug];
        }
        
        return [
            'success' => true,
            'slug' => $this->simpleSlug($prompt)
        ];
    }
    
    /**
     * Simple slug fallback
     */
    private function simpleSlug(string $text): string {
        $slug = strtolower($text);
        $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
        $slug = preg_replace('/[\s-]+/', '-', $slug);
        $slug = trim($slug, '-');
        
        if (strlen($slug) > 50) {
            $slug = substr($slug, 0, 50);
            $slug = preg_replace('/-[^-]*$/', '', $slug);
        }
        
        return $slug ?: 'challenge-' . time();
    }
    
    /**
     * Get vote from AI for an output
     */
    public function getVote(array $voterModel, string $originalPrompt, string $codeToJudge): array {
        $votePrompt = 'You are judging an AI coding challenge. 
The original prompt was: "' . $originalPrompt . '"

Here is the code to evaluate:
```html
' . $codeToJudge . '
```

Rate this submission on 4 criteria (1-10 scale):

1. FUNCTIONALITY: Does it work correctly? Is it bug-free? Does it fulfill the prompt requirements?
2. AESTHETICS: Visual appeal, colors, animations, overall polish and user experience
3. CREATIVITY: Extra features beyond basic requirements? Innovative solutions?
4. CODE_QUALITY: Clean, readable, well-structured, efficient code?

IMPORTANT: Respond with ONLY valid JSON, no markdown, no explanation:
{"functionality":X,"aesthetics":X,"creativity":X,"code_quality":X,"feedback":"One brief sentence"}';

        $result = $this->callAI($voterModel, $votePrompt);
        
        if (!$result['success']) {
            return $result;
        }
        
        // Parse JSON from response
        $content = trim($result['content']);
        
        // Remove markdown code blocks if present
        $content = preg_replace('/^```json?\s*/i', '', $content);
        $content = preg_replace('/\s*```$/', '', $content);
        
        $votes = json_decode($content, true);
        
        if (!$votes || !isset($votes['functionality'])) {
            return [
                'success' => false,
                'error' => 'Invalid vote response format',
                'raw' => $result['content']
            ];
        }
        
        // Validate and clamp values
        foreach (['functionality', 'aesthetics', 'creativity', 'code_quality'] as $key) {
            $votes[$key] = max(1, min(10, (int) ($votes[$key] ?? 5)));
        }
        
        return [
            'success' => true,
            'votes' => $votes,
            'time_ms' => $result['time_ms']
        ];
    }
}
